// LICENSE : MIT
"use strict";
import { moduleInterop } from "@textlint/module-interop";
import fs from "fs";
import path from "path";
// @ts-expect-error
import tryResolve from "try-resolve";
import debug0 from "debug";
import { pathToFileURL } from "node:url";
// formatter
import checkstyleFormatter from "./formatters/checkstyle";
import compactFormatter from "./formatters/compact";
import jslintXMLFormatter from "./formatters/jslint-xml";
import jsonFormatter from "./formatters/json";
import junitFormatter from "./formatters/junit";
import prettyErrorFormatter from "./formatters/pretty-error";
import stylishFormatter from "./formatters/stylish";
import tableFormatter from "./formatters/table";
import tapFormatter from "./formatters/tap";
import unixFormatter from "./formatters/unix";
const builtinFormatterList = {
    checkstyle: checkstyleFormatter,
    compact: compactFormatter,
    "jslint-xml": jslintXMLFormatter,
    json: jsonFormatter,
    junit: junitFormatter,
    "pretty-error": prettyErrorFormatter,
    stylish: stylishFormatter,
    table: tableFormatter,
    tap: tapFormatter,
    unix: unixFormatter
};
const builtinFormatterNames = Object.keys(builtinFormatterList);
// import() can not load Window file path
// convert file path to file URL before import()
// https://github.com/nodejs/node/issues/31710
export async function dynamicImport(targetPath) {
    const fileUrl = pathToFileURL(targetPath).href;
    return import(fileUrl);
}
const debug = debug0("textlint:@textlint/linter-formatter");
export async function loadFormatter(formatterConfig) {
    const formatterName = formatterConfig.formatterName;
    debug(`formatterName: ${formatterName}`);
    if (builtinFormatterNames.includes(formatterName)) {
        return {
            format(results) {
                return builtinFormatterList[formatterName](results, formatterConfig);
            }
        };
    }
    let formatter;
    let formatterPath;
    if (fs.existsSync(formatterName)) {
        formatterPath = formatterName;
    }
    else if (fs.existsSync(path.resolve(process.cwd(), formatterName))) {
        formatterPath = path.resolve(process.cwd(), formatterName);
    }
    else {
        const pkgPath = tryResolve(`textlint-formatter-${formatterName}`) || tryResolve(formatterName);
        if (pkgPath) {
            formatterPath = pkgPath;
        }
    }
    try {
        formatter = moduleInterop((await dynamicImport(formatterPath)).default);
    }
    catch (ex) {
        throw new Error(`Could not find formatter ${formatterName}
${ex}`);
    }
    return {
        format(results) {
            return formatter(results, formatterConfig);
        }
    };
}
/**
 * @deprecated use loadFormatter
 * @param formatterConfig
 */
export function createFormatter(formatterConfig) {
    const formatterName = formatterConfig.formatterName;
    debug(`formatterName: ${formatterName}`);
    if (builtinFormatterNames.includes(formatterName)) {
        return function (results) {
            return builtinFormatterList[formatterName](results, formatterConfig);
        };
    }
    let formatter;
    let formatterPath;
    if (fs.existsSync(formatterName)) {
        formatterPath = formatterName;
    }
    else if (fs.existsSync(path.resolve(process.cwd(), formatterName))) {
        formatterPath = path.resolve(process.cwd(), formatterName);
    }
    else {
        const pkgPath = tryResolve(`textlint-formatter-${formatterName}`) || tryResolve(formatterName);
        if (pkgPath) {
            formatterPath = pkgPath;
        }
    }
    try {
        formatter = moduleInterop(require(formatterPath));
    }
    catch (ex) {
        throw new Error(`Could not find formatter ${formatterName}
${ex}`);
    }
    return function (results) {
        return formatter(results, formatterConfig);
    };
}
export function getFormatterList() {
    return builtinFormatterNames.map((name) => {
        return {
            name
        };
    });
}
//# sourceMappingURL=index.js.map